from __future__ import annotations
import os, csv
from typing import Dict, Any, List

class ExperimentLogger:
    """
    General CSV logger:
    - Default fields cover NASBench301 experiments: seed/method/bench/n_eval/sim_time/elapsed_time/best_score/curr_score/config
    - You can also pass extra_fields to extend other columns
    - Missing fields in row will be automatically filled with empty strings
    """
    DEFAULT_FIELDS = [
        "seed", "method", "bench",
        "n_eval",
        "sim_time",        # Simulated time (from runtime proxy, accumulated)
        "elapsed_time",    # Real elapsed time (wall-clock time from evaluate call)
        "best_score", "curr_score",
        "config",          # To keep consistent with your original log, store as string (dict -> str)
    ]

    def __init__(self, path: str, extra_fields: List[str] | None = None):
        os.makedirs(os.path.dirname(path) or ".", exist_ok=True)
        self.path = path
        self.fields = self.DEFAULT_FIELDS + (extra_fields or [])
        self.f = open(path, "w", newline="")
        self.w = csv.writer(self.f)
        self.w.writerow(self.fields)

    def log(self, row: Dict[str, Any]):
        out = []
        for k in self.fields:
            v = row.get(k, "")
            if k == "config" and isinstance(v, dict):
                v = str(v)  # Simple serialization; for strict parsing you can use json.dumps
            out.append(v)
        self.w.writerow(out)
        self.f.flush()

    def close(self):
        self.f.close()
